/*
 * Decompiled with CFR 0.152.
 */
package io.sentry.util.network;

import io.sentry.ILogger;
import io.sentry.JsonObjectReader;
import io.sentry.SentryLevel;
import io.sentry.util.network.NetworkBody;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class NetworkBodyParser {
    private NetworkBodyParser() {
    }

    @Nullable
    public static NetworkBody fromBytes(@Nullable byte[] bytes, @Nullable String contentType, @Nullable String charset, int maxSizeBytes, @NotNull ILogger logger) {
        if (bytes == null || bytes.length == 0) {
            return null;
        }
        if (contentType != null && NetworkBodyParser.isBinaryContentType(contentType)) {
            return NetworkBody.fromString("[Binary data, " + bytes.length + " bytes, type: " + contentType + "]");
        }
        if (bytes.length > maxSizeBytes) {
            logger.log(SentryLevel.WARNING, "Content exceeds max size limit of " + maxSizeBytes + " bytes", new Object[0]);
            return NetworkBodyParser.createTruncatedNetworkBody(bytes, maxSizeBytes, charset);
        }
        try {
            String effectiveCharset = charset != null ? charset : "UTF-8";
            String content = new String(bytes, effectiveCharset);
            return NetworkBodyParser.parse(content, contentType, logger);
        }
        catch (UnsupportedEncodingException e) {
            logger.log(SentryLevel.WARNING, "Failed to decode bytes: " + e.getMessage(), new Object[0]);
            return NetworkBody.fromString("[Failed to decode bytes, " + bytes.length + " bytes]");
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    private static NetworkBody parse(@Nullable String content, @Nullable String contentType, @Nullable ILogger logger) {
        String trimmed;
        if (content == null) return null;
        if (content.isEmpty()) {
            return null;
        }
        if (contentType != null) {
            String lowerContentType = contentType.toLowerCase();
            if (lowerContentType.contains("application/x-www-form-urlencoded")) {
                return NetworkBodyParser.parseFormUrlEncoded(content, logger);
            }
            if (lowerContentType.contains("xml")) {
                return NetworkBody.fromString(content);
            }
        }
        if (!(trimmed = content.trim()).startsWith("{")) {
            if (!trimmed.startsWith("[")) return NetworkBody.fromString(content);
        }
        try (JsonObjectReader reader = new JsonObjectReader(new StringReader(trimmed));){
            Object parsed = reader.nextObjectOrNull();
            if (parsed instanceof Map) {
                Map map = (Map)parsed;
                NetworkBody networkBody = NetworkBody.fromJsonObject(map);
                return networkBody;
            }
            if (!(parsed instanceof List)) return NetworkBody.fromString(content);
            List list = (List)parsed;
            NetworkBody networkBody = NetworkBody.fromJsonArray(list);
            return networkBody;
        }
        catch (Exception e) {
            if (logger == null) return NetworkBody.fromString(content);
            logger.log(SentryLevel.WARNING, "Failed to parse JSON: " + e.getMessage(), new Object[0]);
        }
        return NetworkBody.fromString(content);
    }

    @Nullable
    private static NetworkBody parseFormUrlEncoded(@NotNull String content, @Nullable ILogger logger) {
        try {
            String[] pairs;
            HashMap<String, Object> params = new HashMap<String, Object>();
            for (String pair : pairs = content.split("&", -1)) {
                String value;
                int idx = pair.indexOf("=");
                if (idx <= 0) continue;
                String key = URLDecoder.decode(pair.substring(0, idx), "UTF-8");
                String string = value = idx < pair.length() - 1 ? URLDecoder.decode(pair.substring(idx + 1), "UTF-8") : "";
                if (params.containsKey(key)) {
                    ArrayList<String> list;
                    Object existing = params.get(key);
                    if (existing instanceof List) {
                        list = (ArrayList<String>)existing;
                        list.add(value);
                        continue;
                    }
                    list = new ArrayList<String>();
                    list.add((String)existing);
                    list.add(value);
                    params.put(key, list);
                    continue;
                }
                params.put(key, value);
            }
            return NetworkBody.fromJsonObject(params);
        }
        catch (UnsupportedEncodingException e) {
            if (logger != null) {
                logger.log(SentryLevel.WARNING, "Failed to parse form data: " + e.getMessage(), new Object[0]);
            }
            return null;
        }
    }

    @NotNull
    private static NetworkBody createTruncatedNetworkBody(@NotNull byte[] bytes, int maxSizeBytes, @Nullable String charset) {
        byte[] truncatedBytes = new byte[maxSizeBytes];
        System.arraycopy(bytes, 0, truncatedBytes, 0, maxSizeBytes);
        try {
            int lastValidIndex;
            String effectiveCharset = charset != null ? charset : "UTF-8";
            String content = new String(truncatedBytes, effectiveCharset);
            for (lastValidIndex = content.length(); lastValidIndex > 0 && content.charAt(lastValidIndex - 1) == '\ufffd'; --lastValidIndex) {
            }
            if (lastValidIndex < content.length()) {
                content = content.substring(0, lastValidIndex);
            }
            content = content + "...[truncated]";
            return NetworkBody.fromString(content);
        }
        catch (UnsupportedEncodingException e) {
            return NetworkBody.fromString("[Failed to decode truncated bytes, " + bytes.length + " bytes]");
        }
    }

    private static boolean isBinaryContentType(@NotNull String contentType) {
        String lower = contentType.toLowerCase();
        return lower.contains("image/") || lower.contains("video/") || lower.contains("audio/") || lower.contains("application/octet-stream") || lower.contains("application/pdf") || lower.contains("application/zip") || lower.contains("application/gzip");
    }
}

