/*
 * Decompiled with CFR 0.152.
 */
package io.ebean.datasource.pool;

import io.ebean.datasource.ConnectionPoolExhaustedException;
import io.ebean.datasource.DataSourceAlert;
import io.ebean.datasource.DataSourceBuilder;
import io.ebean.datasource.DataSourceConfig;
import io.ebean.datasource.DataSourceInitialiseException;
import io.ebean.datasource.DataSourcePool;
import io.ebean.datasource.DataSourcePoolListener;
import io.ebean.datasource.InitDatabase;
import io.ebean.datasource.NewConnectionInitializer;
import io.ebean.datasource.PoolStatus;
import io.ebean.datasource.pool.DriverDataSource;
import io.ebean.datasource.pool.ExecutorFactory;
import io.ebean.datasource.pool.Log;
import io.ebean.datasource.pool.PooledConnection;
import io.ebean.datasource.pool.PooledConnectionQueue;
import io.ebean.datasource.pool.PstmtCache;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Logger;
import javax.sql.DataSource;

final class ConnectionPool
implements DataSourcePool {
    private static final String APPLICATION_NAME = "ApplicationName";
    private final ReentrantLock heartbeatLock = new ReentrantLock(false);
    private final ReentrantLock notifyLock = new ReentrantLock(false);
    private final String name;
    private final AtomicInteger size = new AtomicInteger(0);
    private final DataSourceConfig config;
    private final DataSourceAlert notify;
    private final DataSourcePoolListener poolListener;
    private final NewConnectionInitializer connectionInitializer;
    private final List<String> initSql;
    private final String user;
    private final String schema;
    private final String catalog;
    private final String heartbeatSql;
    private final int heartbeatFreqSecs;
    private final int heartbeatTimeoutSeconds;
    private final int heartbeatMaxPoolExhaustedCount;
    private final long trimPoolFreqMillis;
    private final int transactionIsolation;
    private final boolean autoCommit;
    private final boolean readOnly;
    private final boolean failOnStart;
    private final int maxInactiveMillis;
    private final long validateStaleMillis;
    private final boolean enforceCleanClose;
    private final long maxAgeMillis;
    private final boolean captureStackTrace;
    private final int maxStackTraceSize;
    private final Properties clientInfo;
    private final String applicationName;
    private final DataSource source;
    private final boolean validateOnHeartbeat;
    private long nextTrimTime;
    private final AtomicBoolean dataSourceUp = new AtomicBoolean(false);
    private SQLException dataSourceDownReason;
    private final int minConnections;
    private final int initialConnections;
    private int maxConnections;
    private final int waitTimeoutMillis;
    private final int pstmtCacheSize;
    private final PooledConnectionQueue queue;
    private Heartbeat heartbeat;
    private int heartbeatPoolExhaustedCount;
    private final ExecutorService executor;
    private final long leakTimeMinutes;
    private final LongAdder pscHit = new LongAdder();
    private final LongAdder pscMiss = new LongAdder();
    private final LongAdder pscRem = new LongAdder();
    private final boolean shutdownOnJvmExit;
    private Thread shutdownHook;

    ConnectionPool(String name, DataSourceConfig params) {
        this.config = params;
        this.name = name;
        this.notify = params.getAlert();
        this.poolListener = params.getListener();
        this.connectionInitializer = params.getConnectionInitializer();
        this.autoCommit = params.isAutoCommit();
        this.readOnly = params.isReadOnly();
        this.failOnStart = params.isFailOnStart();
        this.initSql = params.getInitSql();
        this.transactionIsolation = params.getIsolationLevel();
        this.maxInactiveMillis = 1000 * params.getMaxInactiveTimeSecs();
        this.maxAgeMillis = 60000L * (long)params.getMaxAgeMinutes();
        this.leakTimeMinutes = params.getLeakTimeMinutes();
        this.captureStackTrace = params.isCaptureStackTrace();
        this.maxStackTraceSize = params.getMaxStackTraceSize();
        this.pstmtCacheSize = params.getPstmtCacheSize();
        this.minConnections = params.getMinConnections();
        this.initialConnections = params.getInitialConnections();
        this.maxConnections = params.getMaxConnections();
        this.waitTimeoutMillis = params.getWaitTimeoutMillis();
        this.heartbeatFreqSecs = params.getHeartbeatFreqSecs();
        this.heartbeatTimeoutSeconds = params.getHeartbeatTimeoutSeconds();
        this.heartbeatMaxPoolExhaustedCount = params.getHeartbeatMaxPoolExhaustedCount();
        this.heartbeatSql = params.getHeartbeatSql();
        this.validateOnHeartbeat = params.isValidateOnHeartbeat();
        this.trimPoolFreqMillis = 1000L * (long)params.getTrimPoolFreqSecs();
        this.validateStaleMillis = params.validateStaleMillis();
        this.applicationName = params.getApplicationName();
        this.clientInfo = params.getClientInfo();
        this.queue = new PooledConnectionQueue(this);
        this.schema = params.getSchema();
        this.catalog = params.catalog();
        this.user = params.getUsername();
        this.shutdownOnJvmExit = params.isShutdownOnJvmExit();
        this.source = DriverDataSource.of(name, (DataSourceBuilder.Settings)params);
        this.enforceCleanClose = params.enforceCleanClose();
        if (!params.isOffline()) {
            this.init();
        }
        this.nextTrimTime = System.currentTimeMillis() + this.trimPoolFreqMillis;
        this.executor = ExecutorFactory.newExecutor();
    }

    private void init() {
        try {
            if (this.config.useInitDatabase()) {
                this.initialiseDatabase();
            }
            this.initialiseConnections();
        }
        catch (SQLException e) {
            throw new DataSourceInitialiseException("Error initialising DataSource with user: " + this.user + " error:" + e.getMessage(), (Throwable)e);
        }
    }

    void pstmtCacheMetrics(PstmtCache pstmtCache) {
        this.pscHit.add(pstmtCache.hitCount());
        this.pscMiss.add(pstmtCache.missCount());
        this.pscRem.add(pstmtCache.removeCount());
    }

    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        throw new SQLFeatureNotSupportedException("We do not support java.util.logging");
    }

    private void tryEnsureMinimumConnections() {
        this.notifyLock.lock();
        try {
            this.queue.createConnections(this.initialConnections);
            if (this.notify != null) {
                this.notify.dataSourceUp((DataSource)((Object)this));
            }
        }
        catch (SQLException e) {
            Log.error("Error trying to ensure minimum connections, maybe db server is down - message:" + e.getMessage(), e);
        }
        finally {
            this.notifyLock.unlock();
        }
    }

    private void initialiseConnections() throws SQLException {
        long start = System.currentTimeMillis();
        this.dataSourceUp.set(true);
        if (this.failOnStart) {
            this.queue.createConnections(this.initialConnections);
        } else {
            this.tryEnsureMinimumConnections();
        }
        this.startHeartBeatIfStopped();
        if (this.shutdownOnJvmExit && this.shutdownHook == null) {
            this.shutdownHook = new Thread(() -> this.shutdownPool(true, true));
            this.shutdownHook.setName("DataSourcePool-ShutdownHook");
            Runtime.getRuntime().addShutdownHook(this.shutdownHook);
        }
        String ro = this.readOnly ? "readOnly[true] " : "";
        Log.info("DataSource [{0}] {1}autoCommit[{2}] [{3}] min[{4}] max[{5}] in[{6}ms]", this.name, ro, this.autoCommit, ConnectionPool.description(this.transactionIsolation), this.minConnections, this.maxConnections, System.currentTimeMillis() - start, this.validateOnHeartbeat);
    }

    private static String description(int level) {
        switch (level) {
            case 0: {
                return "NONE";
            }
            case 2: {
                return "READ_COMMITTED";
            }
            case 1: {
                return "READ_UNCOMMITTED";
            }
            case 4: {
                return "REPEATABLE_READ";
            }
            case 8: {
                return "SERIALIZABLE";
            }
            case -1: {
                return "NotSet";
            }
        }
        return "UNKNOWN[" + level + "]";
    }

    private void initialiseDatabase() throws SQLException {
        try (Connection connection = this.createConnection();){
            connection.clearWarnings();
        }
        catch (SQLException e) {
            Log.info("Obtaining connection using ownerUsername:{0} to initialise database", this.config.getOwnerUsername());
            try (Connection ownerConnection = this.getConnection(this.config.getOwnerUsername(), this.config.getOwnerPassword());){
                InitDatabase initDatabase = this.config.getInitDatabase();
                initDatabase.run(ownerConnection, this.config);
                ownerConnection.commit();
            }
            catch (SQLException e2) {
                throw new SQLException("Failed to run InitDatabase with ownerUsername:" + this.config.getOwnerUsername() + " message:" + e2.getMessage(), e2);
            }
        }
    }

    public boolean isWrapperFor(Class<?> arg0) {
        return false;
    }

    public <T> T unwrap(Class<T> arg0) throws SQLException {
        throw new SQLException("Not Implemented");
    }

    public String name() {
        return this.name;
    }

    public int size() {
        return this.size.get();
    }

    String schema() {
        return this.schema;
    }

    String catalog() {
        return this.catalog;
    }

    void inc() {
        this.size.incrementAndGet();
    }

    void dec() {
        this.size.decrementAndGet();
    }

    int maxStackTraceSize() {
        return this.maxStackTraceSize;
    }

    public SQLException dataSourceDownReason() {
        return this.dataSourceDownReason;
    }

    private void notifyDataSourceIsDown(SQLException reason) {
        if (this.dataSourceUp.get()) {
            this.reset();
            this.notifyDown(reason);
        }
    }

    private void notifyDown(SQLException reason) {
        this.notifyLock.lock();
        try {
            if (this.dataSourceUp.get()) {
                this.dataSourceUp.set(false);
                this.dataSourceDownReason = reason;
                Log.error("FATAL: DataSource [" + this.name + "] is down or has network error!!!", reason);
                if (this.notify != null) {
                    this.notify.dataSourceDown((DataSource)((Object)this), reason);
                }
            }
        }
        finally {
            this.notifyLock.unlock();
        }
    }

    private void notifyDataSourceIsUp() {
        if (!this.dataSourceUp.get()) {
            this.reset();
            this.notifyUp();
        }
    }

    private void notifyUp() {
        this.notifyLock.lock();
        try {
            if (!this.dataSourceUp.get()) {
                this.dataSourceUp.set(true);
                this.startHeartBeatIfStopped();
                this.dataSourceDownReason = null;
                Log.error("RESOLVED FATAL: DataSource [" + this.name + "] is back up!", new Object[0]);
                if (this.notify != null) {
                    this.notify.dataSourceUp((DataSource)((Object)this));
                }
            } else {
                Log.info("DataSource [{0}] is back up!", this.name);
            }
        }
        finally {
            this.notifyLock.unlock();
        }
    }

    private void trimIdleConnections() {
        if (System.currentTimeMillis() > this.nextTrimTime) {
            try {
                this.queue.trim(this.maxInactiveMillis, this.maxAgeMillis);
                this.nextTrimTime = System.currentTimeMillis() + this.trimPoolFreqMillis;
            }
            catch (Exception e) {
                Log.error("Error trying to trim idle connections - message:" + e.getMessage(), e);
            }
        }
    }

    void heartbeat() {
        this.trimIdleConnections();
        if (this.validateOnHeartbeat) {
            this.testConnection();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testConnection() {
        PooledConnection conn = null;
        try {
            conn = this.getPooledConnection();
            this.heartbeatPoolExhaustedCount = 0;
            if (this.testConnection(conn)) {
                this.notifyDataSourceIsUp();
            } else {
                this.notifyDataSourceIsDown(null);
            }
        }
        catch (ConnectionPoolExhaustedException be) {
            ++this.heartbeatPoolExhaustedCount;
            if (this.heartbeatPoolExhaustedCount > this.heartbeatMaxPoolExhaustedCount) {
                this.notifyDataSourceIsDown((SQLException)((Object)be));
            } else {
                Log.warn("Heartbeat: " + be.getMessage(), new Object[0]);
            }
        }
        catch (SQLException ex) {
            this.notifyDataSourceIsDown(ex);
        }
        finally {
            block30: {
                try {
                    if (conn == null) break block30;
                    try {
                        if (!conn.getAutoCommit()) {
                            conn.rollback();
                        }
                    }
                    finally {
                        conn.closePooledConnection(false);
                    }
                }
                catch (SQLException ex) {
                    Log.warn("Can't close connection in checkDataSource!", new Object[0]);
                }
            }
        }
    }

    private Connection initConnection(Connection conn) throws SQLException {
        if (this.connectionInitializer != null) {
            this.connectionInitializer.preInitialize(conn);
        }
        conn.setAutoCommit(this.autoCommit);
        if (conn.getTransactionIsolation() != this.transactionIsolation) {
            conn.setTransactionIsolation(this.transactionIsolation);
        }
        if (this.readOnly) {
            conn.setReadOnly(true);
        }
        if (this.catalog != null) {
            conn.setCatalog(this.catalog);
        }
        if (this.schema != null) {
            conn.setSchema(this.schema);
        }
        if (this.applicationName != null) {
            try {
                conn.setClientInfo(APPLICATION_NAME, this.applicationName);
            }
            catch (SQLClientInfoException e) {
                Log.error("Error setting clientInfo ApplicationName", e);
            }
        }
        if (this.clientInfo != null) {
            try {
                conn.setClientInfo(this.clientInfo);
            }
            catch (SQLClientInfoException e) {
                Log.error("Error setting clientInfo", e);
            }
        }
        if (this.initSql != null) {
            for (String query : this.initSql) {
                try (Statement stmt = conn.createStatement();){
                    stmt.execute(query);
                }
            }
        }
        if (this.connectionInitializer != null) {
            this.connectionInitializer.postInitialize(conn);
        }
        return conn;
    }

    private Connection createConnection() throws SQLException {
        return this.initConnection(this.source.getConnection());
    }

    public void setMaxSize(int max) {
        this.queue.setMaxSize(max);
        this.maxConnections = max;
    }

    int maxSize() {
        return this.maxConnections;
    }

    int minSize() {
        return this.minConnections;
    }

    int waitTimeoutMillis() {
        return this.waitTimeoutMillis;
    }

    long maxAgeMillis() {
        return this.maxAgeMillis;
    }

    long validateStaleMillis() {
        return this.validateStaleMillis;
    }

    private boolean testConnection(Connection conn) throws SQLException {
        if (this.heartbeatSql == null) {
            return conn.isValid(this.heartbeatTimeoutSeconds);
        }
        try {
            boolean bl;
            block12: {
                Statement stmt = conn.createStatement();
                try {
                    if (this.heartbeatTimeoutSeconds > 0) {
                        stmt.setQueryTimeout(this.heartbeatTimeoutSeconds);
                    }
                    stmt.execute(this.heartbeatSql);
                    bl = true;
                    if (stmt == null) break block12;
                }
                catch (Throwable throwable) {
                    if (stmt != null) {
                        try {
                            stmt.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                stmt.close();
            }
            return bl;
        }
        finally {
            if (!conn.getAutoCommit()) {
                conn.rollback();
            }
        }
    }

    boolean invalidConnection(PooledConnection conn) {
        try {
            return !this.testConnection(conn);
        }
        catch (Exception e) {
            Log.warn("Validation test failed on connection:{0} message: {1}", conn.name(), e.getMessage());
            return true;
        }
    }

    boolean enforceCleanClose() {
        return this.enforceCleanClose;
    }

    void returnConnection(PooledConnection pooledConnection) {
        this.returnTheConnection(pooledConnection, false);
    }

    void returnConnectionForceClose(PooledConnection pooledConnection, boolean testPool) {
        this.returnTheConnection(pooledConnection, true);
        if (testPool) {
            this.testConnection();
        }
    }

    void removeClosedConnection(PooledConnection pooledConnection) {
        this.queue.returnPooledConnection(pooledConnection, true);
    }

    private void returnTheConnection(PooledConnection pooledConnection, boolean forceClose) {
        if (this.poolListener != null && !forceClose) {
            this.poolListener.onBeforeReturnConnection((Connection)pooledConnection);
        }
        this.queue.returnPooledConnection(pooledConnection, forceClose);
    }

    void returnConnectionReset(PooledConnection pooledConnection) {
        this.queue.returnPooledConnection(pooledConnection, true);
        Log.warn("Resetting DataSource on read-only failure [{0}]", this.name);
        this.reset();
    }

    PooledConnection createConnectionForQueue(int connId) throws SQLException {
        try {
            PooledConnection pooledConnection = new PooledConnection(this, connId, this.createConnection());
            pooledConnection.resetForUse();
            this.notifyDataSourceIsUp();
            return pooledConnection;
        }
        catch (SQLException ex) {
            this.notifyDataSourceIsDown(ex);
            throw ex;
        }
    }

    private void reset() {
        this.heartbeatPoolExhaustedCount = 0;
        this.queue.reset(this.leakTimeMinutes);
    }

    public Connection getConnection(String username, String password) throws SQLException {
        return this.initConnection(this.source.getConnection(username, password));
    }

    public Connection getConnection() throws SQLException {
        return this.getPooledConnection();
    }

    private PooledConnection getPooledConnection() throws SQLException {
        PooledConnection c = this.queue.obtainConnection();
        if (this.captureStackTrace) {
            c.setStackTrace(Thread.currentThread().getStackTrace());
        }
        if (this.poolListener != null) {
            this.poolListener.onAfterBorrowConnection((Connection)c);
        }
        return c;
    }

    public void shutdown() {
        this.shutdownPool(true, false);
    }

    public void offline() {
        this.shutdownPool(false, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void shutdownPool(boolean fullShutdown, boolean fromHook) {
        this.heartbeatLock.lock();
        try {
            this.stopHeartBeatIfRunning();
            PoolStatus status = this.queue.shutdown(fullShutdown);
            this.dataSourceUp.set(false);
            if (fullShutdown) {
                this.shutdownExecutor();
            }
            if (fromHook) {
                Log.info("DataSource [{0}] shutdown on JVM exit {1}  psc[hit:{2} miss:{3} rem:{4}]", this.name, status, this.pscHit, this.pscMiss, this.pscRem);
            } else {
                Log.info("DataSource [{0}] shutdown {1}  psc[hit:{2} miss:{3} rem:{4}]", this.name, status, this.pscHit, this.pscMiss, this.pscRem);
                this.removeShutdownHook();
            }
        }
        finally {
            this.heartbeatLock.unlock();
        }
    }

    private void removeShutdownHook() {
        if (this.shutdownHook != null) {
            try {
                Runtime.getRuntime().removeShutdownHook(this.shutdownHook);
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
            this.shutdownHook = null;
        }
    }

    public void online() throws SQLException {
        if (!this.dataSourceUp.get()) {
            this.initialiseConnections();
        }
    }

    public boolean isOnline() {
        return this.dataSourceUp.get();
    }

    public boolean isDataSourceUp() {
        return this.dataSourceUp.get();
    }

    private void startHeartBeatIfStopped() {
        this.heartbeatLock.lock();
        try {
            int freqMillis;
            if (this.heartbeat == null && (freqMillis = this.heartbeatFreqSecs * 1000) > 0) {
                this.heartbeat = ExecutorFactory.newHeartBeat(this, freqMillis);
            }
        }
        finally {
            this.heartbeatLock.unlock();
        }
    }

    private void stopHeartBeatIfRunning() {
        this.heartbeatLock.lock();
        try {
            if (this.heartbeat != null) {
                this.heartbeat.stop();
                this.heartbeat = null;
            }
        }
        finally {
            this.heartbeatLock.unlock();
        }
    }

    void closeConnectionFullyAsync(PooledConnection pc, boolean logErrors) {
        if (!this.executor.isShutdown()) {
            try {
                this.executor.submit(new AsyncCloser(pc, logErrors));
                return;
            }
            catch (RejectedExecutionException e) {
                Log.trace("DataSource [{0}] closing connection synchronously", this.name);
            }
        }
        pc.doCloseConnection(logErrors);
    }

    private void shutdownExecutor() {
        this.executor.shutdown();
        try {
            if (!this.executor.awaitTermination(5L, TimeUnit.SECONDS)) {
                Log.warn("DataSource [{0}] on shutdown, timeout waiting for connections to close", this.name);
            }
        }
        catch (InterruptedException ie) {
            Log.warn("DataSource [{0}] on shutdown, interrupted closing connections", this.name, ie);
        }
        List<Runnable> pendingTasks = this.executor.shutdownNow();
        if (!pendingTasks.isEmpty()) {
            Log.warn("DataSource [{0}] on shutdown, {1} pending connections were not closed", this.name, pendingTasks.size());
        }
    }

    public boolean isAutoCommit() {
        return this.autoCommit;
    }

    public boolean isReadOnly() {
        return this.readOnly;
    }

    int transactionIsolation() {
        return this.transactionIsolation;
    }

    boolean captureStackTrace() {
        return this.captureStackTrace;
    }

    long leakTimeMinutes() {
        return this.leakTimeMinutes;
    }

    int pstmtCacheSize() {
        return this.pstmtCacheSize;
    }

    public int getLoginTimeout() throws SQLException {
        throw new SQLException("Method not supported");
    }

    public void setLoginTimeout(int seconds) throws SQLException {
        throw new SQLException("Method not supported");
    }

    public PrintWriter getLogWriter() {
        return null;
    }

    public void setLogWriter(PrintWriter writer) throws SQLException {
        throw new SQLException("Method not supported");
    }

    public PoolStatus status(boolean reset) {
        return this.queue.status(reset);
    }

    @FunctionalInterface
    static interface Heartbeat {
        public void stop();
    }

    private static final class AsyncCloser
    implements Runnable {
        final PooledConnection pc;
        final boolean logErrors;

        private AsyncCloser(PooledConnection pc, boolean logErrors) {
            this.pc = pc;
            this.logErrors = logErrors;
        }

        @Override
        public void run() {
            this.pc.doCloseConnection(this.logErrors);
        }

        public String toString() {
            return this.pc.toString();
        }
    }

    static final class Status
    implements PoolStatus {
        private final int minSize;
        private final int maxSize;
        private final int free;
        private final int busy;
        private final int waiting;
        private final int highWaterMark;
        private final int waitCount;
        private final int hitCount;
        private final long totalAcquireMicros;
        private final long maxAcquireMicros;
        private final long totalWaitMicros;
        private final long meanAcquireNanos;

        Status(int minSize, int maxSize, int free, int busy, int waiting, int highWaterMark, int waitCount, int hitCount, long totalAcquireNanos, long maxAcquireNanos, long totalWaitNanos) {
            this.minSize = minSize;
            this.maxSize = maxSize;
            this.free = free;
            this.busy = busy;
            this.waiting = waiting;
            this.highWaterMark = highWaterMark;
            this.waitCount = waitCount;
            this.hitCount = hitCount;
            this.totalAcquireMicros = totalAcquireNanos / 1000L;
            this.maxAcquireMicros = maxAcquireNanos / 1000L;
            this.totalWaitMicros = totalWaitNanos / 1000L;
            this.meanAcquireNanos = hitCount == 0 ? 0L : totalAcquireNanos / (long)hitCount;
        }

        public String toString() {
            return "min[" + this.minSize + "] max[" + this.maxSize + "] free[" + this.free + "] busy[" + this.busy + "] waiting[" + this.waiting + "] highWaterMark[" + this.highWaterMark + "] waitCount[" + this.waitCount + "] hitCount[" + this.hitCount + "] totalAcquireMicros[" + this.totalAcquireMicros + "] maxAcquireMicros[" + this.maxAcquireMicros + "] totalWaitMicros[" + this.totalWaitMicros + "]";
        }

        public int minSize() {
            return this.minSize;
        }

        public int maxSize() {
            return this.maxSize;
        }

        public int free() {
            return this.free;
        }

        public int busy() {
            return this.busy;
        }

        public int waiting() {
            return this.waiting;
        }

        public int highWaterMark() {
            return this.highWaterMark;
        }

        public int waitCount() {
            return this.waitCount;
        }

        public int hitCount() {
            return this.hitCount;
        }

        public long totalAcquireMicros() {
            return this.totalAcquireMicros;
        }

        public long totalWaitMicros() {
            return this.totalWaitMicros;
        }

        public long maxAcquireMicros() {
            return this.maxAcquireMicros;
        }

        public long meanAcquireNanos() {
            return this.meanAcquireNanos;
        }
    }
}

